<?php
require __DIR__ . '/../config.php';
set_cors();
if (is_options()) exit(0);

// ─────────────────────────────────────────────────────────────────────────────
// SAFE app-facing endpoint (idempotent-ish).
// Verifies an HMAC-SHA256 signature over canonical JSON of `payload` only.
// If a valid `existing_id` (or `id`) is provided AND its stored payload_json
// exactly matches the new canonical payload, the row is REUSED and
// `expires_at` is EXTENDED (only moves forward). Meta (title/desc/canonical/long)
// is optionally updated if provided.
// Otherwise, a NEW row is created.
// ─────────────────────────────────────────────────────────────────────────────
//
// Request JSON:
// {
//   "v": 1,
//   "payload": { ...SharePayloadV1... },
//   "sig": "<base64url(HMAC-SHA256(canonical(payload)))>",
//   "expiresAt": "2025-11-15T12:00:00Z" | null,
//
//   // Optional metadata
//   "title": "Ask The Word — John 3:16–18",
//   "description": "Open this shared Bible insight in Ask The Word.",
//   "canonical_url": "https://snickitybit.com/h?...",
//   "long_url": "https://snickitybit.com/some/friendly/page",
//
//   // Optional reuse
//   "existing_id": "abcd1234"   // preferred
//   // or
//   "id": "abcd1234"            // accepted for compatibility
// }
// ─────────────────────────────────────────────────────────────────────────────

header('Cache-Control: no-store');
header('X-Content-Type-Options: nosniff');

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  send_json(405, ['error' => 'Method not allowed']);
}

if (!defined('HMAC_SECRET')) {
  send_json(500, ['error' => 'Server HMAC secret not configured']);
}

$hmacKey = base64_decode(HMAC_SECRET, true);
if ($hmacKey === false || strlen($hmacKey) === 0) {
  send_json(500, ['error' => 'Server HMAC secret invalid']);
}

$req = json_input(); // associative array
$ver = isset($req['v']) ? intval($req['v']) : 0;
$payload = $req['payload'] ?? null;
$sig = $req['sig'] ?? '';
$expiresAtIso = array_key_exists('expiresAt', $req) ? $req['expiresAt'] : null;

// Optional metadata (top-level)
$title = isset($req['title']) ? (string)$req['title'] : '';
$desc  = isset($req['description']) ? (string)$req['description'] : '';
$canon = isset($req['canonical_url']) ? (string)$req['canonical_url'] : '';
$long  = isset($req['long_url']) ? (string)$req['long_url'] : '';

// Optional reuse id
$existingId = '';
if (isset($req['existing_id']) && is_string($req['existing_id'])) {
  $existingId = $req['existing_id'];
} elseif (isset($req['id']) && is_string($req['id'])) {
  $existingId = $req['id'];
}

if ($ver !== 1 || !is_array($payload) || !is_string($sig) || $sig === '') {
  send_json(400, ['error' => 'Invalid request']);
}

// Canonical JSON (deterministic keys, stable arrays)
function canonicalize($value) {
  if (is_array($value)) {
    $isList = array_keys($value) === range(0, count($value) - 1);
    if ($isList) {
      $items = [];
      foreach ($value as $v) $items[] = canonicalize($v);
      return '[' . implode(',', $items) . ']';
    } else {
      ksort($value, SORT_STRING);
      $parts = [];
      foreach ($value as $k => $v) {
        $parts[] = json_encode((string)$k, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
                 . ':' . canonicalize($v);
      }
      return '{' . implode(',', $parts) . '}';
    }
  } elseif (is_string($value)) {
    return json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  } elseif (is_int($value) || is_float($value) || is_bool($value) || is_null($value)) {
    return json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  }
  return json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}

$canonical = canonicalize($payload);

// base64url helpers
function b64url_decode($s) {
  $s2 = strtr($s, '-_', '+/');
  $pad = strlen($s2) % 4;
  if ($pad) $s2 .= str_repeat('=', 4 - $pad);
  return base64_decode($s2);
}
function b64url_encode($raw) {
  return rtrim(strtr(base64_encode($raw), '+/', '-_'), '=');
}

// Verify HMAC(SHA-256) of canonical payload JSON
$clientSig = b64url_decode($sig);
if ($clientSig === false) {
  send_json(400, ['error' => 'Bad signature encoding']);
}
$serverSig = hash_hmac('sha256', $canonical, $hmacKey, true);
if (!hash_equals($serverSig, $clientSig)) {
  send_json(401, ['error' => 'Invalid signature']);
}

// Parse optional expiresAt
$requestedExpires = null;  // SQL datetime or null
if ($expiresAtIso !== null) {
  if (!is_string($expiresAtIso) || $expiresAtIso === '') {
    send_json(400, ['error' => 'expiresAt must be ISO-8601 string or null']);
  }
  $requestedExpires = to_sql_datetime($expiresAtIso);
  if ($requestedExpires === null) {
    send_json(400, ['error' => 'Invalid expiresAt format (ISO-8601, e.g. 2025-11-15T12:00:00Z)']);
  }
  if ($requestedExpires <= now_sql()) {
    send_json(400, ['error' => 'expiresAt must be in the future']);
  }
}

// --- helpers for meta normalization ---
function clamp_str(?string $s, int $max): ?string {
  if ($s === null) return null;
  $s = trim($s);
  if ($s === '') return null;
  if (function_exists('mb_substr')) return mb_substr($s, 0, $max);
  return substr($s, 0, $max);
}
function normalize_url(?string $s, int $max): ?string {
  $s = clamp_str($s, $max);
  if ($s === null) return null;
  if (!filter_var($s, FILTER_VALIDATE_URL)) return null;
  if (!preg_match('~^https?://~i', $s)) return null;
  return $s;
}

// Apply schema limits
$titleSafe = clamp_str($title, 200);     // links.title VARCHAR(200)
$descSafe  = clamp_str($desc, 300);      // links.description VARCHAR(300)
$canonSafe = normalize_url($canon, 500); // links.canonical_url VARCHAR(500)
$longSafe  = normalize_url($long, 1000); // links.long_url VARCHAR(1000)

$db = pdo();

// ─────────────────────────────────────────────────────────────────────────────
// REUSE / EXTEND PATH
// ─────────────────────────────────────────────────────────────────────────────
$reuseSucceeded = false;
$reuseId = null;

if ($existingId) {
  // Prefer provided helper if present in your config; otherwise fallback to a simple regex
  $idIsValid = function_exists('is_valid_id') ? is_valid_id($existingId) : (bool)preg_match('~^[A-Za-z0-9]{4,16}$~', $existingId);
  if ($idIsValid) {
    $stmt = $db->prepare('SELECT id, status, payload_json, expires_at, title, description, canonical_url, long_url
                          FROM links WHERE id = ? LIMIT 1');
    $stmt->execute([$existingId]);
    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
      if (($row['status'] ?? '') === 'active') {
        // Only reuse if payload matches exactly
        $samePayload = hash_equals((string)$row['payload_json'], (string)$canonical);

        if ($samePayload) {
          $reuseId = $existingId;

          // Build dynamic UPDATE: extend expires_at forward, update meta if provided
          $sets = [];
          $params = [];

          if ($requestedExpires !== null) {
            $currentExp = $row['expires_at']; // may be NULL
            // Move forward only; never shorten
            $newExp = $requestedExpires;
            if ($currentExp !== null && $currentExp !== '' && $currentExp > $requestedExpires) {
              $newExp = $currentExp;
            }
            if ($currentExp !== $newExp) {
              $sets[] = 'expires_at = ?';
              $params[] = $newExp;
            }
          }

          if ($titleSafe !== null && $titleSafe !== $row['title']) {
            $sets[] = 'title = ?';
            $params[] = $titleSafe;
          }
          if ($descSafe !== null && $descSafe !== $row['description']) {
            $sets[] = 'description = ?';
            $params[] = $descSafe;
          }
          if ($canonSafe !== null && $canonSafe !== $row['canonical_url']) {
            $sets[] = 'canonical_url = ?';
            $params[] = $canonSafe;
          }
          if ($longSafe !== null && $longSafe !== $row['long_url']) {
            $sets[] = 'long_url = ?';
            $params[] = $longSafe;
          }

          if (!empty($sets)) {
            $sql = 'UPDATE links SET ' . implode(', ', $sets) . ' WHERE id = ? AND status = "active"';
            $params[] = $reuseId;
            $u = $db->prepare($sql);
            $u->execute($params);
          }

          // Reuse path succeeded — return SAME id/urls (and existing/new expiresAt if we changed it)
          $reuseSucceeded = true;

          // Fetch latest expires_at if we touched it (or just return whatever we decided)
          $finalExp = $requestedExpires ?? $row['expires_at'];
          send_json(200, [
            'id'        => $reuseId,
            'shortUrl'  => SHORT_DOMAIN . '/s/' . $reuseId,
            'expiresAt' => $finalExp,
            'shareUrl'  => SHORT_DOMAIN . '/share/' . $reuseId,
          ]);
        }
        // else payload mismatch → fall through to NEW row
      }
      // If revoked or non-active, fall through to NEW row
    }
    // If not found, fall through to NEW row
  }
}

// ─────────────────────────────────────────────────────────────────────────────
// NEW ROW INSERT PATH
// ─────────────────────────────────────────────────────────────────────────────

// Generate unique id
function generate_id($db) {
  for ($attempt = 0; $attempt < 5; $attempt++) {
    $candidate = base62_random_id(8); // base62 short id
    $stmt = $db->prepare('SELECT id FROM links WHERE id = ?');
    $stmt->execute([$candidate]);
    if (!$stmt->fetch()) return $candidate;
  }
  return null;
}

$id = generate_id($db);
if ($id === null) {
  send_json(500, ['error' => 'Could not generate id']);
}

// Insert row (includes meta columns)
$stmt = $db->prepare(
  'INSERT INTO links
     (id, payload_json, status, created_at, created_ip, expires_at, clicks, last_access,
      long_url, title, description, canonical_url)
   VALUES
     (?,  ?,            "active", ?,          ?,          ?,          0,      NULL,
      ?,        ?,     ?,           ?)'
);
$stmt->execute([
  $id,
  $canonical,                 // payload_json (canonical)
  now_sql(),                  // created_at (UTC)
  client_ip_bin(),            // created_ip (VARBINARY(16))
  $requestedExpires,          // expires_at (may be NULL)
  $longSafe,                  // long_url
  $titleSafe,                 // title
  $descSafe,                  // description
  $canonSafe                  // canonical_url
]);

send_json(200, [
  'id'        => $id,
  'shortUrl'  => SHORT_DOMAIN . '/s/' . $id,
  'expiresAt' => $requestedExpires,
  'shareUrl'  => SHORT_DOMAIN . '/share/' . $id,
]);
