<?php
// list.php — paginated list of short links with filters
// Requires: ../config.php (pdo(), set_cors(), require_api_key(), send_json())
require __DIR__ . '/../config.php';

ini_set('display_errors', 1);
error_reporting(E_ALL);

set_cors();
if (is_options()) exit(0);

/* ----------------------- Helpers ----------------------- */

function parse_date($s) {
    if (!$s) return null;
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) return null;
    return $s;
}
function col_exists(PDO $db, string $table, string $column): bool {
    try {
        $stmt = $db->prepare(
            "SELECT COUNT(*) 
               FROM information_schema.COLUMNS 
              WHERE table_schema = DATABASE() 
                AND table_name = :t 
                AND column_name = :c"
        );
        $stmt->execute([':t'=>$table, ':c'=>$column]);
        return ((int)$stmt->fetchColumn()) > 0;
    } catch (Throwable $e) {
        return false;
    }
}
function rangeWhere(string $col): string {
    return "$col BETWEEN :from_utc AND :to_utc";
}
function toUtcBounds(?string $fromYmd, ?string $toYmd): array {
    // Defaults: last 30 days inclusive (to = today)
    $todayUTC = new DateTime('now', new DateTimeZone('UTC'));
    $default_to   = $todayUTC->format('Y-m-d');
    $default_from = $todayUTC->sub(new DateInterval('P29D'))->format('Y-m-d');

    $from = $fromYmd ?: $default_from;
    $to   = $toYmd   ?: $default_to;

    $from_dt = new DateTime($from . ' 00:00:00', new DateTimeZone('UTC'));
    $to_dt   = new DateTime($to   . ' 23:59:59', new DateTimeZone('UTC'));

    return [$from, $to, $from_dt->format('Y-m-d H:i:s'), $to_dt->format('Y-m-d H:i:s')];
}

/* ------------------------- Main ------------------------ */

try {
    // API key auth
    require_api_key();

    $db = pdo();

    // Query params
    $page       = max(1, (int)($_GET['page'] ?? 1));
    $page_size  = (int)($_GET['page_size'] ?? 25);
    if ($page_size <= 0 || $page_size > 200) $page_size = 25;

    $statusCsv  = trim((string)($_GET['status'] ?? '')); // e.g. "active,revoked"
    $q          = trim((string)($_GET['q'] ?? ''));      // id prefix/contains
    $fromParam  = parse_date($_GET['from'] ?? '');
    $toParam    = parse_date($_GET['to'] ?? '');
    $sort       = strtolower((string)($_GET['sort'] ?? 'created_at')); // created_at | expires_at | access_count | last_accessed_at | id
    $order      = strtoupper((string)($_GET['order'] ?? 'DESC'));      // ASC | DESC

    // Host for short URL
    $host = $_SERVER['HTTP_HOST'] ?? 'go.snickitybit.com';
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'https';
    $short_base = $scheme . '://' . $host . '/s/';

    // Column availability (we won't SELECT non-existent columns)
    $hasExpiresAt     = col_exists($db, 'links', 'expires_at');
    $hasAccessCount   = col_exists($db, 'links', 'access_count');
    $hasLastAccessed  = col_exists($db, 'links', 'last_accessed_at');
    $hasLongUrl       = col_exists($db, 'links', 'long_url');       // optional
    $hasPayloadJson   = col_exists($db, 'links', 'payload_json');   // optional

    // Build SELECT list
    $selectCols = ['id', 'status', 'created_at'];
    if ($hasExpiresAt)    $selectCols[] = 'expires_at';
    if ($hasAccessCount)  $selectCols[] = 'access_count';
    if ($hasLastAccessed) $selectCols[] = 'last_accessed_at';
    if ($hasLongUrl)      $selectCols[] = 'long_url';
    if ($hasPayloadJson)  $selectCols[] = 'payload_json';

    // Whitelist sort
    $sortable = array_flip(array_merge($selectCols, ['id', 'created_at']));
    if (!isset($sortable[$sort])) $sort = 'created_at';
    if ($order !== 'ASC' && $order !== 'DESC') $order = 'DESC';

    // Filters
    [$fromYmd, $toYmd, $fromUtc, $toUtc] = toUtcBounds($fromParam, $toParam);
    $where = [];
    $bind  = [
        ':from_utc' => $fromUtc,
        ':to_utc'   => $toUtc,
    ];

    // created_at range (always)
    $where[] = rangeWhere('created_at');

    // status filter list
    $statuses = [];
    if ($statusCsv !== '') {
        foreach (explode(',', $statusCsv) as $s) {
            $s = strtolower(trim($s));
            if ($s !== '') $statuses[] = $s;
        }
        if ($statuses) {
            $in = [];
            foreach ($statuses as $i => $s) {
                $ph = ":st_$i";
                $in[] = $ph;
                $bind[$ph] = $s;
            }
            $where[] = "status IN (" . implode(',', $in) . ")";
        }
    }

    // simple "q" filter on id (prefix or contains)
    if ($q !== '') {
        // If looks like exact id, try exact; else prefix/contains
        $bind[':q_like'] = '%' . $q . '%';
        $bind[':q_pref'] = $q . '%';
        $where[] = "(id = :q_exact OR id LIKE :q_pref OR id LIKE :q_like)";
        $bind[':q_exact'] = $q;
    }

    $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

    // COUNT(*) for pagination
    $sqlCount = "SELECT COUNT(*) FROM links $whereSql";
    $stmt = $db->prepare($sqlCount);
    $stmt->execute($bind);
    $total = (int)$stmt->fetchColumn();

    // Page window
    $offset = ($page - 1) * $page_size;

    // Main SELECT
    $sql = "SELECT " . implode(',', $selectCols) . "
              FROM links
            $whereSql
          ORDER BY $sort $order
             LIMIT :lim OFFSET :off";

    $stmt = $db->prepare($sql);
    foreach ($bind as $k => $v) { $stmt->bindValue($k, $v); }
    $stmt->bindValue(':lim', $page_size, PDO::PARAM_INT);
    $stmt->bindValue(':off', $offset, PDO::PARAM_INT);
    $stmt->execute();

    $rows = [];
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $item = [
            'id'         => $r['id'],
            'status'     => $r['status'],
            'created_at' => $r['created_at'],
            'short_url'  => $short_base . $r['id'],
        ];
        if ($hasExpiresAt)    $item['expires_at']      = $r['expires_at'];
        if ($hasAccessCount)  $item['access_count']    = (int)$r['access_count'];
        if ($hasLastAccessed) $item['last_accessed_at']= $r['last_accessed_at'];
        if ($hasLongUrl && !empty($r['long_url'])) {
            $item['long_url'] = $r['long_url'];
        }
        // If you store the canonical 'h' URL or payload JSON, you can expose a preview key here.
        if ($hasPayloadJson && !empty($r['payload_json'])) {
            // keep raw for admin debugging; avoid json_decode to prevent accidental huge payloads
            $item['payload_json_len'] = strlen($r['payload_json']);
        }
        $rows[] = $item;
    }

    $resp = [
        'page'       => $page,
        'page_size'  => $page_size,
        'total'      => $total,
        'has_more'   => ($offset + count($rows)) < $total,
        'range'      => ['from'=>$fromYmd, 'to'=>$toYmd],
        'filters'    => [
            'status' => $statuses,
            'q'      => $q,
            'sort'   => $sort,
            'order'  => $order
        ],
        'capabilities' => [
            'has_expires_at'     => $hasExpiresAt,
            'has_access_count'   => $hasAccessCount,
            'has_last_accessed'  => $hasLastAccessed,
            'has_long_url'       => $hasLongUrl,
            'has_payload_json'   => $hasPayloadJson,
        ],
        'items' => $rows,
    ];

    send_json(200, $resp, ['Cache-Control' => 'no-store']);

} catch (Throwable $e) {
    send_json(500, [
        'error'  => 'internal_error',
        'detail' => $e->getMessage(),
    ], ['Cache-Control' => 'no-store']);
}
