<?php
require __DIR__ . '/../config.php';
set_cors();
if (is_options()) exit(0);
require_api_key(); // Admin-only

// POST is preferred; GET works for manual dry-run.
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
if (!in_array($method, ['GET','POST'], true)) {
    send_json(405, ['error' => 'Method not allowed']);
}

// ---------- Parameters ----------
// Accept from query or JSON body; body wins when both present.
$in = ($method === 'POST') ? json_input() : [];
function read_bool($v, $def=false) {
    if ($v === null) return $def;
    if ($v === true || $v === 1 || $v === '1' || $v === 'true') return true;
    return false;
}
function read_int($v, $def, $min, $max) {
    $n = is_numeric($v) ? intval($v) : $def;
    if ($n < $min) $n = $min;
    if ($n > $max) $n = $max;
    return $n;
}
$olderQ = $_GET['olderThanDays'] ?? null;
$olderB = $in['olderThanDays'] ?? null;
$olderThanDays = read_int($olderB ?? $olderQ, 30, 1, 3650); // default 30d, clamp 1..3650

$limitQ = $_GET['limit'] ?? null;
$limitB = $in['limit'] ?? null;
$limit = read_int($limitB ?? $limitQ, 1000, 1, 5000);

$actionQ = $_GET['action'] ?? null;
$actionB = $in['action'] ?? null;
$action = ($actionB ?? $actionQ) ?: 'delete'; // 'delete' | 'mark'
if ($action !== 'delete' && $action !== 'mark') {
    send_json(400, ['error' => 'Invalid action; use delete or mark']);
}

$dryQ = $_GET['dryRun'] ?? null;
$dryB = $in['dryRun'] ?? null;
$dryRun = read_bool($dryB ?? $dryQ, true); // default: dry-run

// Optional status filter (e.g., only 'revoked'); usually not needed.
// Leave empty to include any status except already 'purged'.
$statusQ = $_GET['status'] ?? null;
$statusB = $in['status'] ?? null;
$statusFilter = is_string($statusB ?? $statusQ) ? trim(($statusB ?? $statusQ)) : '';

// ---------- Work ----------
$db = pdo();

// Build WHERE:
// expired, not already purged, and expired more than N days ago.
$where = 'expires_at IS NOT NULL
          AND expires_at < UTC_TIMESTAMP()
          AND expires_at < DATE_SUB(UTC_TIMESTAMP(), INTERVAL :older DAY)
          AND (status IS NULL OR status <> "purged")';

$params = [':older' => $olderThanDays];

if ($statusFilter !== '') {
    $where .= ' AND status = :status';
    $params[':status'] = $statusFilter;
}

// First, SELECT candidate IDs (limited) so we can report counts and sample.
$sqlIds = 'SELECT id FROM links WHERE ' . $where . ' ORDER BY expires_at ASC LIMIT :lim';
$stmt = $db->prepare($sqlIds);
foreach ($params as $k => $v) {
    if ($k === ':older') $stmt->bindValue($k, $v, PDO::PARAM_INT);
    else $stmt->bindValue($k, $v);
}
$stmt->bindValue(':lim', $limit, PDO::PARAM_INT);
$stmt->execute();
$ids = $stmt->fetchAll(PDO::FETCH_COLUMN);

// Count total matching (without limit) for visibility.
$sqlCount = 'SELECT COUNT(*) AS c FROM links WHERE ' . $where;
$countStmt = $db->prepare($sqlCount);
foreach ($params as $k => $v) {
    if ($k === ':older') $countStmt->bindValue($k, $v, PDO::PARAM_INT);
    else $countStmt->bindValue($k, $v);
}
$countStmt->execute();
$totalMatching = (int)($countStmt->fetch()['c'] ?? 0);

$affected = 0;

if (!$dryRun && !empty($ids)) {
    if ($action === 'delete') {
        // Hard delete in-batch
        // Use an IN (…) list. Chunk if needed.
        $chunks = array_chunk($ids, 500);
        foreach ($chunks as $chunk) {
            $ph = implode(',', array_fill(0, count($chunk), '?'));
            $del = $db->prepare("DELETE FROM links WHERE id IN ($ph)");
            $del->execute($chunk);
            $affected += $del->rowCount();
        }
    } else { // mark
        $chunks = array_chunk($ids, 500);
        foreach ($chunks as $chunk) {
            $ph = implode(',', array_fill(0, count($chunk), '?'));
            $upd = $db->prepare("UPDATE links SET status = 'purged' WHERE id IN ($ph)");
            $upd->execute($chunk);
            $affected += $upd->rowCount();
        }
    }
}

// Response
send_json(200, [
    'ok' => true,
    'dryRun' => $dryRun,
    'action' => $action,
    'olderThanDays' => $olderThanDays,
    'limit' => $limit,
    'statusFilter' => ($statusFilter === '' ? null : $statusFilter),
    'totalMatching' => $totalMatching,       // all rows that qualify (ignores limit)
    'considered' => count($ids),             // rows selected this run (<= limit)
    'affected' => $affected,                 // rows actually changed this run
    'sampleIds' => array_slice($ids, 0, 10), // a peek at what matched
]);
