<?php
require __DIR__ . '/../config.php';

// ---- SETTINGS ---------------------------------------------------------------
$LOG_DIR  = dirname(__DIR__) . '/logs';                // ../logs next to /api
$LOG_FILE = $LOG_DIR . '/purge_expired.log';
$ENFORCE_CLI = false;  // set true to block HTTP; otherwise allow with token below
$REQUIRE_TOKEN = false; // set true to require ?token=... over HTTP
$HTTP_TOKEN = getenv('PURGE_TOKEN') ?: ''; // set in cPanel → Cron Env or .htaccess/php.ini
$REVOKED_KEEP_DAYS = 7; // how long to keep revoked rows before purging
// ----------------------------------------------------------------------------

// Optional request protection
$isCli = (php_sapi_name() === 'cli');
if ($ENFORCE_CLI && !$isCli) {
    http_response_code(403);
    echo json_encode(['ok' => false, 'error' => 'forbidden (CLI only)']);
    exit;
}
if (!$isCli && $REQUIRE_TOKEN) {
    $token = $_GET['token'] ?? '';
    if (!is_string($token) || $token === '' || $HTTP_TOKEN === '' || !hash_equals($HTTP_TOKEN, $token)) {
        http_response_code(403);
        echo json_encode(['ok' => false, 'error' => 'forbidden (bad token)']);
        exit;
    }
}

// Prepare logging
function ensure_log_dir($dir) {
    if (!is_dir($dir)) @mkdir($dir, 0755, true);
}
function log_line($file, $msg) {
    $ts = gmdate('Y-m-d H:i:s'); // UTC
    @file_put_contents($file, "[$ts] $msg\n", FILE_APPEND | LOCK_EX);
}
ensure_log_dir($LOG_DIR);

// Start
$start = microtime(true);
$err = null;
$expiredCount = 0;
$revokedCount = 0;

try {
    $db = pdo();

    // 1) Purge expired active links (expires_at < NOW())
    $stmt1 = $db->prepare('
        DELETE FROM links
        WHERE status = "active"
          AND expires_at IS NOT NULL
          AND expires_at < UTC_TIMESTAMP()
    ');
    $stmt1->execute();
    $expiredCount = $stmt1->rowCount();

    // 2) Purge revoked links older than N days
    $stmt2 = $db->prepare('
        DELETE FROM links
        WHERE status = "revoked"
          AND created_at < (UTC_TIMESTAMP() - INTERVAL :days DAY)
    ');
    $stmt2->execute([':days' => $REVOKED_KEEP_DAYS]);
    $revokedCount = $stmt2->rowCount();

} catch (Throwable $e) {
    $err = $e->getMessage();
    http_response_code(500);
}

// Finish & log
$durMs = (int) round((microtime(true) - $start) * 1000);
$mem   = memory_get_peak_usage(true);

if ($err === null) {
    log_line($LOG_FILE, "OK expired=$expiredCount revoked=$revokedCount dur=${durMs}ms mem=${mem}");
    header('Content-Type: application/json');
    echo json_encode([
        'ok' => true,
        'expired_deleted' => $expiredCount,
        'revoked_deleted' => $revokedCount,
        'duration_ms' => $durMs,
        'timestamp' => gmdate('c')
    ]);
} else {
    log_line($LOG_FILE, "ERROR \"$err\" dur=${durMs}ms mem=${mem}");
    header('Content-Type: application/json');
    echo json_encode([
        'ok' => false,
        'error' => $err,
        'duration_ms' => $durMs,
        'timestamp' => gmdate('c')
    ]);
}
