<?php
require __DIR__ . '/../config.php';
set_cors();
if (is_options()) exit(0);

// ─────────────────────────────────────────────────────────────────────────────
// SAFE app-facing endpoint (idempotent-ish) with dedupe + share counting.
//
// Rules:
// - Signature is over canonical JSON of `payload` ONLY.
// - If an active row exists with the exact same canonical payload:
//      * Reuse that row,
//      * Increment shared_count,
//      * last_shared_at = now,
//      * Extend expires_at FORWARD (never shorten),
//      * Optionally update meta (title/desc/canonical/long) if provided.
// - If `existing_id` refers to an active row with EXACT same payload:
//      * Same as above (increment + extend + meta).
// - Else create a new row with shared_count=1, last_shared_at=now.
// ─────────────────────────────────────────────────────────────────────────────
//
// Request JSON:
// {
//   "v": 1,
//   "payload": { ...SharePayloadV1... },
//   "sig": "<base64url(HMAC-SHA256(canonical(payload)))>",
//   "expiresAt": "2025-11-15T12:00:00Z" | null,
//
//   // Optional metadata
//   "title": "...",
//   "description": "...",
//   "canonical_url": "https://...",
//   "long_url": "https://...",
//
//   // Optional reuse hint
//   "existing_id": "abcd1234" | "id": "abcd1234"
// }
// ─────────────────────────────────────────────────────────────────────────────

header('Cache-Control: no-store');
header('X-Content-Type-Options: nosniff');

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  send_json(405, ['error' => 'Method not allowed']);
}

if (!defined('HMAC_SECRET')) {
  send_json(500, ['error' => 'Server HMAC secret not configured']);
}

$hmacKey = base64_decode(HMAC_SECRET, true);
if ($hmacKey === false || strlen($hmacKey) === 0) {
  send_json(500, ['error' => 'Server HMAC secret invalid']);
}

$req = json_input(); // associative array
$ver = isset($req['v']) ? intval($req['v']) : 0;
$payload = $req['payload'] ?? null;
$sig = $req['sig'] ?? '';
$expiresAtIso = array_key_exists('expiresAt', $req) ? $req['expiresAt'] : null;

// Optional metadata (top-level)
$title = isset($req['title']) ? (string)$req['title'] : '';
$desc  = isset($req['description']) ? (string)$req['description'] : '';
$canon = isset($req['canonical_url']) ? (string)$req['canonical_url'] : '';
$long  = isset($req['long_url']) ? (string)$req['long_url'] : '';

// Optional reuse id
$existingId = '';
if (isset($req['existing_id']) && is_string($req['existing_id'])) {
  $existingId = $req['existing_id'];
} elseif (isset($req['id']) && is_string($req['id'])) {
  $existingId = $req['id'];
}

if ($ver !== 1 || !is_array($payload) || !is_string($sig) || $sig === '') {
  send_json(400, ['error' => 'Invalid request']);
}

// Canonical JSON (deterministic keys, stable arrays)
function canonicalize($value) {
  if (is_array($value)) {
    $isList = array_keys($value) === range(0, count($value) - 1);
    if ($isList) {
      $items = [];
      foreach ($value as $v) $items[] = canonicalize($v);
      return '[' . implode(',', $items) . ']';
    } else {
      ksort($value, SORT_STRING);
      $parts = [];
      foreach ($value as $k => $v) {
        $parts[] = json_encode((string)$k, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
                 . ':' . canonicalize($v);
      }
      return '{' . implode(',', $parts) . '}';
    }
  } elseif (is_string($value)) {
    return json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  } elseif (is_int($value) || is_float($value) || is_bool($value) || is_null($value)) {
    return json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  }
  return json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}

$canonical = canonicalize($payload);

// base64url helpers
function b64url_decode($s) {
  $s2 = strtr($s, '-_', '+/');
  $pad = strlen($s2) % 4;
  if ($pad) $s2 .= str_repeat('=', 4 - $pad);
  return base64_decode($s2);
}
function b64url_encode($raw) {
  return rtrim(strtr(base64_encode($raw), '+/', '-_'), '=');
}

// Verify HMAC(SHA-256) of canonical payload JSON
$clientSig = b64url_decode($sig);
if ($clientSig === false) {
  send_json(400, ['error' => 'Bad signature encoding']);
}
$serverSig = hash_hmac('sha256', $canonical, $hmacKey, true);
if (!hash_equals($serverSig, $clientSig)) {
  send_json(401, ['error' => 'Invalid signature']);
}

// Parse optional expiresAt
$requestedExpires = null;  // SQL datetime or null
if ($expiresAtIso !== null) {
  if (!is_string($expiresAtIso) || $expiresAtIso === '') {
    send_json(400, ['error' => 'expiresAt must be ISO-8601 string or null']);
  }
  $requestedExpires = to_sql_datetime($expiresAtIso);
  if ($requestedExpires === null) {
    send_json(400, ['error' => 'Invalid expiresAt format (ISO-8601, e.g. 2025-11-15T12:00:00Z)']);
  }
  if ($requestedExpires <= now_sql()) {
    send_json(400, ['error' => 'expiresAt must be in the future']);
  }
}

// --- helpers for meta normalization ---
function clamp_str(?string $s, int $max): ?string {
  if ($s === null) return null;
  $s = trim($s);
  if ($s === '') return null;
  if (function_exists('mb_substr')) return mb_substr($s, 0, $max);
  return substr($s, 0, $max);
}
function normalize_url(?string $s, int $max): ?string {
  $s = clamp_str($s, $max);
  if ($s === null) return null;
  if (!filter_var($s, FILTER_VALIDATE_URL)) return null;
  if (!preg_match('~^https?://~i', $s)) return null;
  return $s;
}

// Apply schema limits
$titleSafe = clamp_str($title, 200);     // links.title VARCHAR(200)
$descSafe  = clamp_str($desc, 300);      // links.description VARCHAR(300)
$canonSafe = normalize_url($canon, 500); // links.canonical_url VARCHAR(500)
$longSafe  = normalize_url($long, 1000); // links.long_url VARCHAR(1000)

$db = pdo();

// Column guards
$hasSharedCount  = col_exists($db, 'links', 'shared_count');
$hasLastSharedAt = col_exists($db, 'links', 'last_shared_at');
$hasPayloadHash  = col_exists($db, 'links', 'payload_sha'); // optional optimization

// Utility: extend expires forward + update meta (AND increment share)
function apply_reuse_update(PDO $db, string $id, ?string $requestedExpires, ?string $titleSafe, ?string $descSafe, ?string $canonSafe, ?string $longSafe, bool $hasSharedCount, bool $hasLastSharedAt) {
  // Build dynamic UPDATE
  $sets = [];
  $params = [];

  if ($requestedExpires !== null) {
    // Use LEAST/CASE: keep the later of current vs requested (never shorten)
    $sets[] = 'expires_at = CASE WHEN expires_at IS NULL OR expires_at < ? THEN ? ELSE expires_at END';
    $params[] = $requestedExpires;
    $params[] = $requestedExpires;
  }
  if ($titleSafe !== null) { $sets[] = 'title = ?';          $params[] = $titleSafe; }
  if ($descSafe  !== null) { $sets[] = 'description = ?';    $params[] = $descSafe;  }
  if ($canonSafe !== null) { $sets[] = 'canonical_url = ?';  $params[] = $canonSafe; }
  if ($longSafe  !== null) { $sets[] = 'long_url = ?';       $params[] = $longSafe;  }

  // Always increment share on reuse
  if ($hasSharedCount)  { $sets[] = 'shared_count = shared_count + 1'; }
  if ($hasLastSharedAt) { $sets[] = 'last_shared_at = UTC_TIMESTAMP()'; }

  if (!empty($sets)) {
    $sql = 'UPDATE links SET ' . implode(', ', $sets) . ' WHERE id = ? AND status = "active"';
    $params[] = $id;
    $u = $db->prepare($sql);
    $u->execute($params);
  }
}

// ─────────────────────────────────────────────────────────────────────────────
// REUSE PATH A: explicit existing_id (payload must match)
// ─────────────────────────────────────────────────────────────────────────────
if ($existingId) {
  $idIsValid = function_exists('is_valid_id') ? is_valid_id($existingId) : (bool)preg_match('~^[A-Za-z0-9]{4,16}$~', $existingId);
  if ($idIsValid) {
    $stmt = $db->prepare('SELECT id, status, payload_json, expires_at, title, description, canonical_url, long_url
                          FROM links WHERE id = ? LIMIT 1');
    $stmt->execute([$existingId]);
    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
      if (($row['status'] ?? '') === 'active') {
        $samePayload = hash_equals((string)$row['payload_json'], (string)$canonical);
        if ($samePayload) {
          apply_reuse_update($db, $existingId, $requestedExpires, $titleSafe, $descSafe, $canonSafe, $longSafe, $hasSharedCount, $hasLastSharedAt);

          $finalExp = $requestedExpires ?? $row['expires_at'];
          send_json(200, [
            'id'        => $existingId,
            'shortUrl'  => SHORT_DOMAIN . '/s/' . $existingId,
            'expiresAt' => $finalExp,
            'shareUrl'  => SHORT_DOMAIN . '/share/' . $existingId,
          ]);
        }
      }
    }
    // if not found / not active / payload mismatch → fall through
  }
}

// ─────────────────────────────────────────────────────────────────────────────
// REUSE PATH B: dedupe by payload (match existing active row by canonical payload)
// Try payload_sha first (if column exists). If not found, FALL BACK to payload_json.
// This handles older rows whose payload_sha is NULL.
// ─────────────────────────────────────────────────────────────────────────────
$matchedId = null;
$existingExpires = null;

if ($hasPayloadHash) {
  // 1) Try hash match
  $hashRaw = hash('sha256', $canonical, true); // raw 32 bytes for VARBINARY(32)
  $stmt = $db->prepare('SELECT id, expires_at FROM links WHERE status="active" AND payload_sha = ? LIMIT 1');
  $stmt->execute([$hashRaw]);
  if ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $matchedId = $r['id'];
    $existingExpires = $r['expires_at'];
  }

  // 2) If not found via hash, fall back to exact payload_json match
  if ($matchedId === null) {
    $stmt = $db->prepare('SELECT id, expires_at FROM links WHERE status="active" AND payload_json = ? LIMIT 1');
    $stmt->execute([$canonical]);
    if ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
      $matchedId = $r['id'];
      $existingExpires = $r['expires_at'];

      // Optional: backfill payload_sha for this matched row so future lookups are O(1)
      try {
        $bf = $db->prepare('UPDATE links SET payload_sha = ? WHERE id = ? AND payload_sha IS NULL');
        $bf->execute([$hashRaw, $matchedId]);
      } catch (Throwable $e) { /* ignore */ }
    }
  }
} else {
  // No payload_sha column → use payload_json
  $stmt = $db->prepare('SELECT id, expires_at FROM links WHERE status="active" AND payload_json = ? LIMIT 1');
  $stmt->execute([$canonical]);
  if ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $matchedId = $r['id'];
    $existingExpires = $r['expires_at'];
  }
}

if ($matchedId !== null) {
  apply_reuse_update($db, $matchedId, $requestedExpires, $titleSafe, $descSafe, $canonSafe, $longSafe, $hasSharedCount, $hasLastSharedAt);
  $finalExp = $requestedExpires ?? ($existingExpires ?? null);
  send_json(200, [
    'id'        => $matchedId,
    'shortUrl'  => SHORT_DOMAIN . '/s/' . $matchedId,
    'expiresAt' => $finalExp,
    'shareUrl'  => SHORT_DOMAIN . '/share/' . $matchedId,
  ]);
}

// ─────────────────────────────────────────────────────────────────────────────
// NEW ROW INSERT PATH
// ─────────────────────────────────────────────────────────────────────────────

function generate_id($db) {
  for ($attempt = 0; $attempt < 5; $attempt++) {
    $candidate = base62_random_id(8);
    $stmt = $db->prepare('SELECT id FROM links WHERE id = ?');
    $stmt->execute([$candidate]);
    if (!$stmt->fetch()) return $candidate;
  }
  return null;
}

$id = generate_id($db);
if ($id === null) {
  send_json(500, ['error' => 'Could not generate id']);
}

// Build insert column list (with guards)
// BEFORE building $cols/$vals/$params:
$cols = [
  'id', 'payload_json', 'status', 'created_at', 'created_ip',
  'expires_at', 'clicks', 'last_access',
  'long_url', 'title', 'description', 'canonical_url'
];
$vals = [
  '?',  '?',            '"active"', 'UTC_TIMESTAMP()', ' ? ',
  ' ? ',     '0',     'NULL',
  ' ? ',     ' ? ',   ' ? ',        ' ? '
];
$params = [
  $id,
  $canonical,
  client_ip_bin(),
  $requestedExpires,
  $longSafe,
  $titleSafe,
  $descSafe,
  $canonSafe
];

// ADD (if hasSharedCount / hasLastSharedAt already present in your file, keep them)
if ($hasSharedCount)  { $cols[] = 'shared_count';  $vals[] = '1'; }
if ($hasLastSharedAt) { $cols[] = 'last_shared_at';$vals[] = 'UTC_TIMESTAMP()'; }

// FIX: use payload_sha (raw binary)
if ($hasPayloadHash) {
  $cols[] = 'payload_sha';
  $vals[] = ' ? ';
  $params[] = hash('sha256', $canonical, true); // raw 32-byte hash
}


$sql = 'INSERT INTO links (' . implode(',', $cols) . ') VALUES (' . implode(',', $vals) . ')';
$stmt = $db->prepare($sql);
$stmt->execute($params);

send_json(200, [
  'id'        => $id,
  'shortUrl'  => SHORT_DOMAIN . '/s/' . $id,
  'expiresAt' => $requestedExpires,
  'shareUrl'  => SHORT_DOMAIN . '/share/' . $id,
]);

/* ---------- helpers ---------- */

function col_exists(PDO $db, string $table, string $column): bool {
  try {
    $stmt = $db->prepare(
      "SELECT COUNT(*) FROM information_schema.COLUMNS
       WHERE table_schema = DATABASE() AND table_name = :t AND column_name = :c"
    );
    $stmt->execute([':t'=>$table, ':c'=>$column]);
    return ((int)$stmt->fetchColumn()) > 0;
  } catch (Throwable $e) { return false; }
}
