<?php
// list.php — paginated list of short links with filters
// Requires: ../config.php (pdo(), set_cors(), require_api_key(), send_json())
require __DIR__ . '/../config.php';

ini_set('display_errors', 1);
error_reporting(E_ALL);

set_cors();
if (is_options()) exit(0);

/* ----------------------- Helpers ----------------------- */

function parse_date($s) {
    if (!$s) return null;
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) return null;
    return $s;
}
function col_exists(PDO $db, string $table, string $column): bool {
    try {
        $stmt = $db->prepare(
            "SELECT COUNT(*) 
               FROM information_schema.COLUMNS 
              WHERE table_schema = DATABASE() 
                AND table_name = :t 
                AND column_name = :c"
        );
        $stmt->execute([':t'=>$table, ':c'=>$column]);
        return ((int)$stmt->fetchColumn()) > 0;
    } catch (Throwable $e) {
        return false;
    }
}
function rangeWhere(string $col): string {
    return "$col BETWEEN :from_utc AND :to_utc";
}
function toUtcBounds(?string $fromYmd, ?string $toYmd): array {
    // Defaults: last 30 days inclusive (to = today)
    $todayUTC = new DateTime('now', new DateTimeZone('UTC'));
    $default_to   = $todayUTC->format('Y-m-d');
    $default_from = $todayUTC->sub(new DateInterval('P29D'))->format('Y-m-d');

    $from = $fromYmd ?: $default_from;
    $to   = $toYmd   ?: $default_to;

    $from_dt = new DateTime($from . ' 00:00:00', new DateTimeZone('UTC'));
    $to_dt   = new DateTime($to   . ' 23:59:59', new DateTimeZone('UTC'));

    return [$from, $to, $from_dt->format('Y-m-d H:i:s'), $to_dt->format('Y-m-d H:i:s')];
}

/* ------------------------- Main ------------------------ */

try {
    // API key auth
    require_api_key();

    $db = pdo();

    // Query params
    $page       = max(1, (int)($_GET['page'] ?? 1));
    $page_size  = (int)($_GET['page_size'] ?? 25);
    if ($page_size <= 0 || $page_size > 200) $page_size = 25;

    $statusCsv  = trim((string)($_GET['status'] ?? '')); // e.g. "active,revoked"
    $q          = trim((string)($_GET['q'] ?? ''));      // id prefix/contains
    $fromParam  = parse_date($_GET['from'] ?? '');
    $toParam    = parse_date($_GET['to'] ?? '');
    $sort       = strtolower((string)($_GET['sort'] ?? 'created_at')); // created_at | expires_at | clicks | last_access | shared_count | last_shared_at | id | title
    $order      = strtoupper((string)($_GET['order'] ?? 'DESC'));      // ASC | DESC

    // Host for short URL
    $host = $_SERVER['HTTP_HOST'] ?? parse_url(SHORT_DOMAIN, PHP_URL_HOST) ?? 'go.snickitybit.com';
    $scheme = 'https';
    $short_base = $scheme . '://' . $host . '/s/';

    // Column availability
    $hasExpiresAt     = col_exists($db, 'links', 'expires_at');
    $hasClicks        = col_exists($db, 'links', 'clicks');
    $hasLastAccess    = col_exists($db, 'links', 'last_access');
    $hasSharedCount   = col_exists($db, 'links', 'shared_count');
    $hasLastSharedAt  = col_exists($db, 'links', 'last_shared_at');
    $hasLongUrl       = col_exists($db, 'links', 'long_url');
    $hasPayloadJson   = col_exists($db, 'links', 'payload_json');
    $hasTitle         = col_exists($db, 'links', 'title');

    // Build SELECT list
    $selectCols = ['id', 'status', 'created_at'];
    if ($hasTitle)        $selectCols[] = 'title';
    if ($hasExpiresAt)    $selectCols[] = 'expires_at';
    if ($hasClicks)       $selectCols[] = 'clicks';
    if ($hasLastAccess)   $selectCols[] = 'last_access';
    if ($hasSharedCount)  $selectCols[] = 'shared_count';
    if ($hasLastSharedAt) $selectCols[] = 'last_shared_at';
    if ($hasLongUrl)      $selectCols[] = 'long_url';
    if ($hasPayloadJson)  $selectCols[] = 'payload_json';

    // Whitelist sort
    $sortable = array_flip(array_merge($selectCols, ['id', 'created_at']));
    if (!isset($sortable[$sort])) $sort = 'created_at';
    if ($order !== 'ASC' && $order !== 'DESC') $order = 'DESC';

    // Filters
    [$fromYmd, $toYmd, $fromUtc, $toUtc] = toUtcBounds($fromParam, $toParam);
    $where = [];
    $bind  = [
        ':from_utc' => $fromUtc,
        ':to_utc'   => $toUtc,
    ];

    // created_at range (always)
    $where[] = rangeWhere('created_at');

    // status filter list
    $statuses = [];
    if ($statusCsv !== '') {
        foreach (explode(',', $statusCsv) as $s) {
            $s = strtolower(trim($s));
            if ($s !== '') $statuses[] = $s;
        }
        if ($statuses) {
            $in = [];
            foreach ($statuses as $i => $s) {
                $ph = ":st_$i";
                $in[] = $ph;
                $bind[$ph] = $s;
            }
            $where[] = "status IN (" . implode(',', $in) . ")";
        }
    }

    // simple "q" filter on id (prefix or contains)
    if ($q !== '') {
        $bind[':q_like'] = '%' . $q . '%';
        $bind[':q_pref'] = $q . '%';
        $where[] = "(id = :q_exact OR id LIKE :q_pref OR id LIKE :q_like)";
        $bind[':q_exact'] = $q;
    }

    $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

    // COUNT(*) for pagination
    $sqlCount = "SELECT COUNT(*) FROM links $whereSql";
    $stmt = $db->prepare($sqlCount);
    $stmt->execute($bind);
    $total = (int)$stmt->fetchColumn();

    // Page window
    $offset = ($page - 1) * $page_size;

    // Main SELECT
    $sql = "SELECT " . implode(',', $selectCols) . "
              FROM links
            $whereSql
          ORDER BY $sort $order
             LIMIT :lim OFFSET :off";

    $stmt = $db->prepare($sql);
    foreach ($bind as $k => $v) { $stmt->bindValue($k, $v); }
    $stmt->bindValue(':lim', $page_size, PDO::PARAM_INT);
    $stmt->bindValue(':off', $offset, PDO::PARAM_INT);
    $stmt->execute();

    $rows = [];
    while ($r = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $item = [
            'id'         => $r['id'],
            'status'     => $r['status'],
            'created_at' => $r['created_at'],
            'short_url'  => $short_base . $r['id'],
        ];
        if ($hasTitle)        $item['title']          = $r['title'];
        if ($hasExpiresAt)    $item['expires_at']     = $r['expires_at'];
        if ($hasClicks)       $item['clicks']         = isset($r['clicks']) ? (int)$r['clicks'] : null;
        if ($hasLastAccess)   $item['last_access']    = $r['last_access'];
        if ($hasSharedCount)  $item['shared_count']   = isset($r['shared_count']) ? (int)$r['shared_count'] : null;
        if ($hasLastSharedAt) $item['last_shared_at'] = $r['last_shared_at'];
        if ($hasLongUrl && !empty($r['long_url'])) {
            $item['long_url'] = $r['long_url'];
        }
        if ($hasPayloadJson && !empty($r['payload_json'])) {
            $item['payload_json_len'] = strlen($r['payload_json']);
        }
        $rows[] = $item;
    }

    $resp = [
        'page'       => $page,
        'page_size'  => $page_size,
        'total'      => $total,
        'has_more'   => ($offset + count($rows)) < $total,
        'range'      => ['from'=>$fromYmd, 'to'=>$toYmd],
        'filters'    => [
            'status' => $statuses,
            'q'      => $q,
            'sort'   => $sort,
            'order'  => $order
        ],
        'capabilities' => [
            'has_expires_at'     => $hasExpiresAt,
            'has_clicks'         => $hasClicks,
            'has_last_access'    => $hasLastAccess,
            'has_shared_count'   => $hasSharedCount,
            'has_last_shared_at' => $hasLastSharedAt,
            'has_long_url'       => $hasLongUrl,
            'has_payload_json'   => $hasPayloadJson,
        ],
        'items' => $rows,
    ];

    send_json(200, $resp, ['Cache-Control' => 'no-store']);

} catch (Throwable $e) {
    send_json(500, [
        'error'  => 'internal_error',
        'detail' => $e->getMessage(),
    ], ['Cache-Control' => 'no-store']);
}
