<?php
require __DIR__ . '/../config.php';
set_cors();
if (is_options()) exit(0);
require_api_key(); // Admin-only

// POST is preferred; GET works for manual dry-run.
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
if (!in_array($method, ['GET','POST'], true)) {
    send_json(405, ['error' => 'Method not allowed']);
}

// ---------- Helpers ----------
function read_bool($v, $def=false) {
    if ($v === null) return $def;
    if ($v === true || $v === 1 || $v === '1' || $v === 'true' || $v === 'on') return true;
    return false;
}
function read_int($v, $def, $min, $max) {
    $n = is_numeric($v) ? intval($v) : $def;
    if ($n < $min) $n = $min;
    if ($n > $max) $n = $max;
    return $n;
}

// ---------- Parameters (body wins) ----------
$in = ($method === 'POST') ? json_input() : [];

$olderQ = $_GET['olderThanDays'] ?? null;
$olderB = $in['olderThanDays'] ?? null;
$olderThanDays = read_int($olderB ?? $olderQ, 30, 0, 3650); // allow 0 now

$limitQ = $_GET['limit'] ?? null;
$limitB = $in['limit'] ?? null;
$limit = read_int($limitB ?? $limitQ, 1000, 1, 50000);

$actionQ = $_GET['action'] ?? null;
$actionB = $in['action'] ?? null;
$action = ($actionB ?? $actionQ) ?: 'delete'; // 'delete' | 'mark'

$dryQ = $_GET['dryRun'] ?? null;
$dryB = $in['dryRun'] ?? null;
$dryRun = read_bool($dryB ?? $dryQ, true); // default: dry-run

$statusQ = $_GET['status'] ?? null;
$statusB = $in['status'] ?? null;
$statusFilter = is_string($statusB ?? $statusQ) ? trim(($statusB ?? $statusQ)) : '';

// NEW: all=true to mark every non-purged link (ignores expires_at/limit)
$allQ = $_GET['all'] ?? null;
$allB = $in['all'] ?? null;
$applyAll = read_bool($allB ?? $allQ, false);
if ($applyAll) {
    $action = 'mark'; // force mark for safety
}

// ---------- Work ----------
$db = pdo();

$affected = 0;
$totalMatching = 0;
$ids = [];

// Build WHERE depending on "all"
$params = [];
if ($applyAll) {
    // All rows except already purged; optional status narrowing.
    $where = '(status IS NULL OR status <> "purged")';
    if ($statusFilter !== '') {
        $where .= ' AND status = :status';
        $params[':status'] = $statusFilter;
    }

    // Count everything that would be affected.
    $sqlCount = 'SELECT COUNT(*) AS c FROM links WHERE ' . $where;
    $countStmt = $db->prepare($sqlCount);
    foreach ($params as $k => $v) {
        $countStmt->bindValue($k, $v);
    }
    $countStmt->execute();
    $totalMatching = (int)($countStmt->fetch()['c'] ?? 0);

    // For visibility, sample a few IDs (no limit on update, but sample 10).
    $sqlIds = 'SELECT id FROM links WHERE ' . $where . ' ORDER BY created_at ASC LIMIT 10';
    $stmt = $db->prepare($sqlIds);
    foreach ($params as $k => $v) {
        $stmt->bindValue($k, $v);
    }
    $stmt->execute();
    $ids = $stmt->fetchAll(PDO::FETCH_COLUMN);

    if (!$dryRun && $totalMatching > 0) {
        // Update in one statement (faster) — mark everything not already purged.
        $sqlUpd = 'UPDATE links SET status = "purged" WHERE ' . $where;
        $upd = $db->prepare($sqlUpd);
        foreach ($params as $k => $v) {
            $upd->bindValue($k, $v);
        }
        $upd->execute();
        $affected = $upd->rowCount();
    }
} else {
    // Legacy: only expired older than N days, not already purged; optional status.
    $where = 'expires_at IS NOT NULL
              AND expires_at < DATE_SUB(UTC_TIMESTAMP(), INTERVAL :older DAY)
              AND (status IS NULL OR status <> "purged")';
    $params[':older'] = $olderThanDays;

    if ($statusFilter !== '') {
        $where .= ' AND status = :status';
        $params[':status'] = $statusFilter;
    }

    // Count (no limit)
    $sqlCount = 'SELECT COUNT(*) AS c FROM links WHERE ' . $where;
    $countStmt = $db->prepare($sqlCount);
    foreach ($params as $k => $v) {
        if ($k === ':older') $countStmt->bindValue($k, (int)$v, PDO::PARAM_INT);
        else $countStmt->bindValue($k, $v);
    }
    $countStmt->execute();
    $totalMatching = (int)($countStmt->fetch()['c'] ?? 0);

    // SELECT IDs (with limit) for sampling and batch ops
    $sqlIds = 'SELECT id FROM links WHERE ' . $where . ' ORDER BY expires_at ASC LIMIT ' . $limit;
    $stmt = $db->prepare($sqlIds);
    foreach ($params as $k => $v) {
        if ($k === ':older') $stmt->bindValue($k, (int)$v, PDO::PARAM_INT);
        else $stmt->bindValue($k, $v);
    }
    $stmt->execute();
    $ids = $stmt->fetchAll(PDO::FETCH_COLUMN);

    if (!$dryRun && !empty($ids)) {
        if ($action === 'delete') {
            // Hard delete in chunks
            $chunks = array_chunk($ids, 500);
            foreach ($chunks as $chunk) {
                $ph = implode(',', array_fill(0, count($chunk), '?'));
                $del = $db->prepare("DELETE FROM links WHERE id IN ($ph)");
                $del->execute($chunk);
                $affected += $del->rowCount();
            }
        } else {
            // Mark in chunks
            $chunks = array_chunk($ids, 500);
            foreach ($chunks as $chunk) {
                $ph = implode(',', array_fill(0, count($chunk), '?'));
                $upd = $db->prepare("UPDATE links SET status = 'purged' WHERE id IN ($ph)");
                $upd->execute($chunk);
                $affected += $upd->rowCount();
            }
        }
    }
}

// Response
send_json(200, [
    'ok' => true,
    'dryRun' => $dryRun,
    'action' => $action,
    'mode' => ($applyAll ? 'all' : 'expired'),
    'olderThanDays' => ($applyAll ? null : $olderThanDays),
    'limit' => ($applyAll ? null : $limit),
    'statusFilter' => ($statusFilter === '' ? null : $statusFilter),
    'totalMatching' => $totalMatching,
    'considered' => $applyAll ? $totalMatching : count($ids),
    'affected' => $affected,
    'sampleIds' => array_slice($ids, 0, 10),
]);
