<?php
require __DIR__ . '/../config.php';

// Input
$id    = $_GET['id'] ?? '';
$payload = null;

// Default meta
$title = 'Shared Insight — Ask His Word';
$desc  = 'Tap to open in the app. Includes verse, question, and explanation.';

// App / URL constants
$appStoreId = '6754179331';
$base       = rtrim(SHORT_DOMAIN, '/');

// Universal Link base and a click variant
$ulBase  = $base . '/s/' . rawurlencode($id);
$ulClick = $ulBase . '?open=1&ts=' . time();

// Optional custom URL scheme
$customScheme = defined('APP_CUSTOM_SCHEME') ? trim(APP_CUSTOM_SCHEME) : '';
$schemeUrl    = $customScheme !== '' ? ($customScheme . '://s/' . rawurlencode($id)) : '';

// App Store
$appStore = 'https://apps.apple.com/us/app/ask-the-word/id' . $appStoreId;

// Preview assets (ok to leave on main site)
$iconUrl  = 'https://snickitybit.com/assets/asktheword-icon.png';
$cardUrl  = 'https://snickitybit.com/assets/asktheword-card.png';

// ─────────────────────────────────────────────────────────────────────────────
// Helpers
// ─────────────────────────────────────────────────────────────────────────────
function h(?string $s): string {
    return htmlspecialchars($s ?? '', ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

// Inline markdown: **bold**, *italic*, `code`
function md_inline(string $s): string {
    // Escape once
    $s = htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
    // code
    $s = preg_replace('/`([^`]+)`/u', '<code>$1</code>', $s);
    // bold
    $s = preg_replace('/\*\*([^\*]+)\*\*/u', '<strong>$1</strong>', $s);
    // italic (single * … *)
    $s = preg_replace('/(?<!\*)\*([^\*]+)\*(?!\*)/u', '<em>$1</em>', $s);
    return $s;
}

/**
 * Minimal block-level markdown for the app’s style:
 * - Lines starting with "## " → <h2>
 * - Lines starting with "### " → <h3>
 * - Lines starting with "> " (contiguous) → <blockquote><p>…</p></blockquote>
 * - Paragraphs separated by blank lines
 * - Preserves single newlines as <br> within paragraphs/quotes
 */
function render_markdown_min(string $txt): string {
    $lines = preg_split('/\R/u', $txt);
    $out   = [];
    $para  = [];
    $quote = [];

    $flush_para = function() use (&$out, &$para) {
        if (!empty($para)) {
            $html = md_inline(implode("\n", $para));
            $html = nl2br($html);
            $out[] = "<p>{$html}</p>";
            $para = [];
        }
    };
    $flush_quote = function() use (&$out, &$quote) {
        if (!empty($quote)) {
            $joined = md_inline(implode("\n", $quote));
            $joined = nl2br($joined);
            $out[] = "<blockquote><p>{$joined}</p></blockquote>";
            $quote = [];
        }
    };

    foreach ($lines as $raw) {
        $line = rtrim($raw, "\r");

        // Headings (priority: ### before ##)
        if (preg_match('/^\s*###\s+(.*)$/u', $line, $m)) {
            $flush_quote(); $flush_para();
            $out[] = '<h3>' . md_inline($m[1]) . '</h3>';
            continue;
        }
        if (preg_match('/^\s*##\s+(.*)$/u', $line, $m)) {
            $flush_quote(); $flush_para();
            $out[] = '<h2>' . md_inline($m[1]) . '</h2>';
            continue;
        }

        // Blockquote
        if (preg_match('/^\s*>\s?(.*)$/u', $line, $m)) {
            // We're in quote mode; flush any paragraph first
            $flush_para();
            $quote[] = $m[1];
            continue;
        } else {
            // If we were collecting a quote, a non-quote line ends it
            $flush_quote();
        }

        // Blank line -> end paragraph
        if (trim($line) === '') {
            $flush_para();
        } else {
            $para[] = $line;
        }
    }

    // Flush any trailing blocks
    $flush_quote();
    $flush_para();

    return implode("\n", $out);
}

/** Build "John 3:16-18" string from payload['ref'] */
function ref_string(?array $ref): string {
    if (!$ref) return '';
    $book = trim((string)($ref['book'] ?? ''));
    if ($book === '') return '';
    $chap = isset($ref['chapter']) ? (int)$ref['chapter'] : null;
    $sv   = isset($ref['startVerse']) ? (int)$ref['startVerse'] : null;
    $ev   = isset($ref['endVerse']) ? (int)$ref['endVerse'] : null;

    $s = $book;
    if ($chap) {
        $s .= ' ' . $chap;
        if ($sv) {
            $s .= ':' . $sv;
            if ($ev && $ev !== $sv) $s .= '-' . $ev;
        }
    }
    return $s;
}

// ─────────────────────────────────────────────────────────────────────────────
// Fetch payload & set nicer title/desc
// ─────────────────────────────────────────────────────────────────────────────
if (is_valid_id($id)) {
    try {
        $db = pdo();
        $stmt = $db->prepare(
          'SELECT payload_json,
                  (expires_at IS NULL OR expires_at > UTC_TIMESTAMP()) AS not_expired,
                  status
           FROM links WHERE id = ? LIMIT 1'
        );
        $stmt->execute([$id]);
        if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            if (($row['status'] ?? '') === 'active' && (int)($row['not_expired'] ?? 0) === 1) {
                $payload = json_decode((string)$row['payload_json'], true);
                if (is_array($payload)) {
                    $refStr = ref_string($payload['ref'] ?? null);
                    if ($refStr) $title = $refStr . ' — Shared Insight';
                    if (!empty($payload['question'])) $desc = (string)$payload['question'];
                }
            }
        }
    } catch (Throwable $e) { /* ignore */ }
}

// ─────────────────────────────────────────────────────────────────────────────
// UA detection (prefer custom scheme for environments that suppress UL handoff)
// ─────────────────────────────────────────────────────────────────────────────
$ua = strtolower($_SERVER['HTTP_USER_AGENT'] ?? '');
$isChrome = strpos($ua, 'crios') !== false || (strpos($ua, 'chrome/') !== false && strpos($ua, 'edgios') === false && strpos($ua, 'fxios') === false);
$isGmail  = strpos($ua, 'gmail') !== false;
$isFB     = strpos($ua, 'fbav') !== false || strpos($ua, 'fban') !== false || strpos($ua, 'facebook') !== false;
$isIG     = strpos($ua, 'instagram') !== false;

$preferScheme  = ($schemeUrl !== '') && ($isChrome || $isGmail || $isFB || $isIG);
$primaryHref   = $preferScheme ? $schemeUrl : $ulClick;
$primaryLabel  = 'Open in Ask His Word';
// Removed secondary/UL button per request.
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <title><?= h($title) ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />

  <!-- Smart App Banner -->
  <meta name="apple-itunes-app" content="app-id=<?= h($appStoreId) ?>, app-argument=<?= h($ulBase) ?>">

  <!-- Open Graph / Twitter -->
  <meta property="og:title" content="<?= h($title) ?>">
  <meta property="og:description" content="<?= h($desc) ?>">
  <meta property="og:image" content="<?= h($cardUrl) ?>">
  <meta property="og:url" content="<?= h($ulBase) ?>">
  <meta name="twitter:card" content="summary_large_image">

  <!-- App Links hints -->
  <meta property="al:ios:url" content="<?= h($ulBase) ?>">
  <meta property="al:ios:app_store_id" content="<?= h($appStoreId) ?>">
  <meta property="al:ios:app_name" content="Ask His Word">

  <style>
    :root {
      --bg: #fafafa;
      --fg: #1f2937; /* slate-800 */
      --muted: #6b7280; /* slate-500 */
      --card: #ffffff;
      --sep: #e5e7eb; /* gray-200 */
      --chip: #f3f4f6; /* gray-100 */
      --accent: #0a84ff;
      --accent-ghost: #eff6ff;
      --shadow: 0 10px 30px rgba(0,0,0,.06);
      --radius: 14px;
    }
    * { box-sizing: border-box; }
    body { font-family: ui-sans-serif, system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial, sans-serif;
           margin:0; padding:24px; background:var(--bg); color:var(--fg); }
    .wrap { max-width: 860px; margin: 0 auto; }
    .card { background:var(--card); border-radius: var(--radius); padding: 22px 20px; box-shadow: var(--shadow); }

    .head { display:flex; align-items: baseline; gap: 12px; flex-wrap: wrap; }
    .ref  { font-size: 20px; font-weight: 700; }
    .meta { font-size: 12px; color: var(--muted); display:flex; gap:8px; align-items:center; }

    .timestamp { font-size: 12px; color: var(--muted); margin-top: 4px; }

    .row   { margin-top: 16px; }
    .label { font-size: 13px; font-weight: 600; color: var(--muted); margin-bottom: 6px; }
    .text  { font-size: 16px; line-height: 1.55; }

    .answer { margin-top: 10px; }
    .answer .text p { margin: 0 0 10px; }
    .answer code { padding: 0 4px; border-radius: 4px; background: var(--chip); }

    .settings { margin-top: 12px; background: #f8fafc; border: 1px solid var(--sep);
                border-radius: 12px; padding: 12px; }
    .settings .grid { display:grid; grid-template-columns: 1fr 1fr; gap: 8px 16px; }
    .metric { display:flex; gap:10px; }
    .metric .k { color: var(--muted); font-weight: 600; font-size: 13px; }
    .metric .v { font-size: 13px; }

    hr { border:0; border-top: 1px solid var(--sep); margin: 16px 0; }

    .btns-top { display:flex; gap:10px; margin: 12px 0 8px; flex-wrap: wrap; }
    .btn  { appearance:none; border:0; padding:12px 16px; border-radius:10px; cursor:pointer;
            font-weight:600; text-decoration:none; }
    .primary { background:var(--accent); color:#fff; }
    .ghost   { background:var(--chip); color:#111; }

    .hint { color:#6b7280; font-size:12px; margin-top:10px; }
  </style>
</head>
<body>
  <div class="wrap">
    <div class="card">

      <?php if ($payload): ?>
        <?php
          // Build header fields like SharedInsightView / AIHistoryDetailView
          $refStr       = ref_string($payload['ref'] ?? null);
          $translation  = (string)($payload['translation'] ?? ($payload['translationCode'] ?? ''));
          $source       = (string)($payload['source'] ?? ($payload['bibleSource'] ?? ''));
          $question     = (string)($payload['question'] ?? '');
          $answerRaw    = (string)($payload['answer'] ?? ($payload['responseFull'] ?? $payload['responsePreview'] ?? ''));
          $answerHtml   = render_markdown_min($answerRaw);

          // Optional meta
          $mode     = $payload['mode'] ?? null;
          $audience = $payload['audience'] ?? null;
          $tone     = $payload['tone'] ?? null;
          $depth    = $payload['depth'] ?? null;
          $prompt   = $payload['promptKey'] ?? null;
          $adv      = isset($payload['isAdvanced']) ? ($payload['isAdvanced'] ? 'Yes' : 'No') : null;

          // Optional timestamp if present in payload
          $timestamp = $payload['timestamp'] ?? null;
        ?>

        <!-- Header -->
        <div class="head">
          <div class="ref"><?= h($refStr ?: 'Shared Insight') ?></div>
          <?php if ($translation || $source): ?>
            <div class="meta">
              <?php if ($translation): ?><span><?= h($translation) ?></span><?php endif; ?>
              <?php if ($translation && $source): ?><span>•</span><?php endif; ?>
              <?php if ($source): ?><span><?= h($source) ?></span><?php endif; ?>
            </div>
          <?php endif; ?>
        </div>
        <?php if ($timestamp): ?>
          <div class="timestamp"><?= h($timestamp) ?></div>
        <?php endif; ?>

        <!-- Buttons moved to the TOP (under header) -->
        <div class="btns-top">
          <a class="btn primary" href="<?= h($primaryHref) ?>" rel="external"><?= h($primaryLabel) ?></a>
          <a class="btn ghost" href="<?= h($appStore) ?>">Get the App</a>
        </div>

        <hr>

        <!-- Question -->
        <?php if ($question !== ''): ?>
          <div class="row">
            <div class="label">Question</div>
            <div class="text"><?= nl2br(h($question)) ?></div>
          </div>
        <?php endif; ?>

        <!-- Answer -->
        <div class="row answer">
          <div class="label">Answer</div>
          <div class="text"><?= $answerHtml ?></div>
        </div>

        <!-- Settings / Meta -->
        <?php if ($mode || $audience || $tone || $depth || $prompt || $adv || $refStr || $translation || $source): ?>
          <div class="row settings">
            <div class="label" style="margin-bottom:8px;">Settings</div>
            <div class="grid">
              <?php if ($mode):        ?><div class="metric"><div class="k">Mode</div><div class="v"><?= h($mode) ?></div></div><?php endif; ?>
              <?php if ($audience):    ?><div class="metric"><div class="k">Audience</div><div class="v"><?= h($audience) ?></div></div><?php endif; ?>
              <?php if ($tone):        ?><div class="metric"><div class="k">Tone</div><div class="v"><?= h($tone) ?></div></div><?php endif; ?>
              <?php if ($depth):       ?><div class="metric"><div class="k">Depth</div><div class="v"><?= h($depth) ?></div></div><?php endif; ?>
              <?php if ($prompt):      ?><div class="metric"><div class="k">Prompt</div><div class="v"><?= h($prompt) ?></div></div><?php endif; ?>
              <?php if ($adv):         ?><div class="metric"><div class="k">Model</div><div class="v"><?= h($adv) ?></div></div><?php endif; ?>
              <?php if ($refStr):      ?><div class="metric"><div class="k">Reference</div><div class="v"><?= h($refStr) ?></div></div><?php endif; ?>
              <?php if ($translation): ?><div class="metric"><div class="k">Translation</div><div class="v"><code><?= h($translation) ?></code></div></div><?php endif; ?>
              <?php if ($source):      ?><div class="metric"><div class="k">Bible Source</div><div class="v"><code><?= h($source) ?></code></div></div><?php endif; ?>
            </div>
          </div>
        <?php endif; ?>

      <?php else: ?>
        <div class="head"><div class="ref">Shared Insight</div></div>
        <div class="btns-top">
          <a class="btn primary" href="<?= h($primaryHref) ?>" rel="external">Open in Ask His Word</a>
          <a class="btn ghost" href="<?= h($appStore) ?>">Get the App</a>
        </div>
        <hr>
        <p class="text">This shared item may be expired, revoked, or invalid.</p>
      <?php endif; ?>

      <p class="hint">
        If the app doesn’t open, you may be in an in-app browser or your default browser is Chrome.
        Tap the ••• menu and choose <strong>Open in Safari</strong>, then try again.
      </p>
    </div>
  </div>
</body>
</html>
