<?php
// Dynamic OG image generator for Ask His Word
// /share/og/<id>.png -> share_og.php?id=<id>
// Shows a BIG Book/Chapter:Verse title, blue gradient, faint logo, and badges.

ini_set('display_errors', 1);
error_reporting(E_ALL);

require __DIR__ . '/config.php';

// ------------ Config ------------
const OG_WIDTH  = 1200;
const OG_HEIGHT = 630;

// Assets
$LOGO_PATH = __DIR__ . '/assets/logo.png';

// Fonts (match your actual filenames)
$FONT_BOLD = __DIR__ . '/assets/fonts/MerriweatherVF.ttf';
$FONT_SEMI = __DIR__ . '/assets/fonts/InterVF.ttf';
$FONT_REG  = __DIR__ . '/assets/fonts/InterVF.ttf';


// ------------ Entry ------------
try {
    $id = $_GET['id'] ?? '';
    if (!function_exists('is_valid_id') || !is_valid_id($id)) {
        return og_fallback("Ask His Word", "invalid_id");
    }

    $db = pdo();
    $stmt = $db->prepare("SELECT status, expires_at, payload_json FROM links WHERE id=? LIMIT 1");
    $stmt->execute([$id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        return og_fallback("Not Found", "no_row");
    }

    $status = strtolower($row['status'] ?? 'active');
    $expired = !empty($row['expires_at']) ? (strtotime($row['expires_at'].' UTC') <= time()) : false;
    if ($status !== 'active' || $expired) {
        return og_fallback("Link Unavailable", "inactive_or_expired");
    }

    $payload = json_decode((string)$row['payload_json'], true);
    if (!is_array($payload)) $payload = [];

    $ref = format_ref($payload);
    if ($ref === '') $ref = 'Ask His Word';

    return og_render($ref, $LOGO_PATH, $FONT_BOLD, $FONT_SEMI, $FONT_REG);

} catch (Throwable $e) {
    return og_fallback("Ask His Word", "exception");
}

// ------------ Helpers ------------
function format_ref(array $p): string {
    $book  = $p['ref']['book']     ?? $p['book']     ?? '';
    $chap  = $p['ref']['chapter']  ?? $p['chapter']  ?? null;
    $v0    = $p['ref']['startVerse'] ?? $p['ref']['verseStart'] ?? $p['verseStart'] ?? null;
    $v1    = $p['ref']['endVerse']   ?? $p['ref']['verseEnd']   ?? $p['verseEnd']   ?? null;

    $book = trim((string)$book);
    $out = $book !== '' ? $book : '';
    if (!empty($chap) && is_numeric($chap)) {
        $out .= ($out ? ' ' : '') . intval($chap);
        if (!empty($v0) && is_numeric($v0) && intval($v0) > 0) {
            $out .= ':' . intval($v0);
            if (!empty($v1) && is_numeric($v1) && intval($v1) > 0 && intval($v1) != intval($v0)) {
                $out .= '-' . intval($v1);
            }
        }
    }
    return trim($out);
}

// ------------ Renderers ------------
function og_render(string $ref, string $logoPath, string $fontBold, string $fontSemi, string $fontReg) {
    $w = OG_WIDTH; $h = OG_HEIGHT;

    $im = imagecreatetruecolor($w, $h);
    imagesavealpha($im, true);

    // Background gradient (#2563EB -> #1D4ED8)
    for ($y = 0; $y < $h; $y++) {
        $r = (int)(37 + ($y / $h) * (29 - 37));
        $g = (int)(99 + ($y / $h) * (78 - 99));
        $b = (int)(235 + ($y / $h) * (216 - 235));
        $color = imagecolorallocate($im, $r, $g, $b);
        imageline($im, 0, $y, $w, $y, $color);
    }

    // Watermark logo (faint)
    if (file_exists($logoPath)) {
        $logo = @imagecreatefrompng($logoPath);
        if ($logo) {
            $lw = imagesx($logo); $lh = imagesy($logo);
            $targetW = 720; $targetH = 720;
            $dst = imagecreatetruecolor($targetW, $targetH);
            imagesavealpha($dst, true);
            $trans = imagecolorallocatealpha($dst, 0, 0, 0, 127);
            imagefill($dst, 0, 0, $trans);
            imagecopyresampled($dst, $logo, 0, 0, 0, 0, $targetW, $targetH, $lw, $lh);
            imagecopymerge($im, $dst, (int)(($w - $targetW) / 2), (int)(($h - $targetH) / 2), 0, 0, $targetW, $targetH, 12);
            imagedestroy($dst); imagedestroy($logo);
        }
    }

    // Colors
    $white   = imagecolorallocate($im, 255,255,255);
    $accent  = imagecolorallocate($im, 219,234,254);
    $badgeBg = imagecolorallocatealpha($im, 255,255,255, 100);

    // Detect TTF availability
   $ttfFns = function_exists('imagettftext') && function_exists('imagettfbbox');
    $fontBoldExists = file_exists($fontBold);
    $fontSemiExists = file_exists($fontSemi);
    $fontRegExists  = file_exists($fontReg);
    
    // We only really need bold + one body font:
    $hasTTF = $ttfFns && $fontBoldExists && ($fontSemiExists || $fontRegExists);
    
    // Debug why
    header('X-ATW-OG: ttfFns='.($ttfFns?'1':'0')
        .';bold='.($fontBoldExists?'1':'0')
        .';semi='.($fontSemiExists?'1':'0')
        .';reg='.($fontRegExists?'1':'0')
    .';useTTF='.($hasTTF?'1':'0'));

   

    if ($hasTTF) {
        // BIG title auto-fit to width (one line)
        $marginX = 80;
        $maxWidth = $w - 2 * $marginX;

        // Start huge and shrink until it fits
        $size = 220;               // much larger than before
        $min  = 44;
        while ($size >= $min) {
            $bbox = imagettfbbox($size, 0, $fontBold, $ref);
            $tw = $bbox[2] - $bbox[0];
            if ($tw <= $maxWidth) break;
            $size -= 2;
        }
        $bbox = imagettfbbox($size, 0, $fontBold, $ref);
        $tw = $bbox[2] - $bbox[0];
        $th = $bbox[1] - $bbox[7];

        $x = (int)(($w - $tw) / 2);
        $y = (int)($h * 0.56);     // a touch below center

        imagettftext($im, $size, 0, $x, $y, $white, $fontBold, $ref);

        // Badges
        draw_badge_ttf($im, 50, 72, "Ask His Word", 28, $fontSemi, $badgeBg, $accent);
        draw_badge_ttf($im, $w - 50, $h - 60, "Open in App", 24, $fontReg, $badgeBg, $accent, 'right');

    } else {
        // Bitmap fallback (always small – upload TTFs to get large type!)
        header('X-ATW-OG-Note: bitmap-fallback');
        $font = 5;
        $tw = imagefontwidth($font) * strlen($ref);
        imagestring($im, $font, max(20, (int)(($w - $tw)/2)), (int)($h*0.52), $ref, $white);
        draw_badge_bitmap($im, 34, 60, "Ask His Word");
        draw_badge_bitmap($im, $w - 34, $h - 60, "Open in App", 'right');
    }

    header('Content-Type: image/png');
    header('Cache-Control: public, max-age=600');
    imagepng($im);
    imagedestroy($im);
    return true;
}

function draw_badge_ttf($im, int $x, int $y, string $text, int $size, string $font, $bg, $fg, string $align='left') {
    $padH = 18; $padV = 10; $radius = 18;
    $b = imagettfbbox($size, 0, $font, $text);
    $w = ($b[2]-$b[0]) + 2*$padH;
    $h = ($b[1]-$b[7]) + 2*$padV;

    if ($align === 'right') $x = $x - $w;
    $yTop = $y - $h + 4;

    // pill
    imagefilledrectangle($im, $x+$radius, $yTop, $x+$w-$radius, $yTop+$h, $bg);
    imagefilledellipse($im, $x+$radius, $yTop+$radius, 2*$radius, 2*$radius, $bg);
    imagefilledellipse($im, $x+$w-$radius, $yTop+$radius, 2*$radius, 2*$radius, $bg);

    imagettftext($im, $size, 0, $x + $padH, $yTop + $h - $padV - 4, $fg, $font, $text);
}

function draw_badge_bitmap($im, int $x, int $y, string $text, string $align='left') {
    $font = 3;
    $padH = 8; $padV = 6;
    $w = imagefontwidth($font) * strlen($text) + 2*$padH;
    $h = imagefontheight($font) + 2*$padV;
    if ($align === 'right') $x = $x - $w;
    $yTop = $y - $h + 4;

    $bg = imagecolorallocatealpha($im, 255,255,255, 100);
    $fg = imagecolorallocate($im, 219,234,254);

    imagefilledrectangle($im, $x, $yTop, $x+$w, $yTop+$h, $bg);
    imagestring($im, $font, $x+$padH, $yTop+$padV, $text, $fg);
}

function og_fallback(string $text, string $why="fallback") {
    header('X-ATW-OG: fallback=' . $why);
    $w = OG_WIDTH; $h = OG_HEIGHT;
    if (!extension_loaded('gd')) {
        header('Content-Type: image/png');
        echo base64_decode('iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVQIW2P4//8/AwAI/AL+7aKkHwAAAABJRU5ErkJggg==');
        return true;
    }
    $im = imagecreatetruecolor($w, $h);
    $bg = imagecolorallocate($im, 37,99,235);
    imagefilledrectangle($im, 0, 0, $w, $h, $bg);
    $white = imagecolorallocate($im, 255,255,255);
    $font = 5;
    $tw = imagefontwidth($font) * strlen($text);
    imagestring($im, $font, (int)(($w - $tw)/2), (int)(($h - 16)/2), $text, $white);
    header('Content-Type: image/png');
    header('Cache-Control: public, max-age=300');
    imagepng($im);
    imagedestroy($im);
    return true;
}
